import math
import cmath

import ipc_constants

fs = 48000

bessel_coeff = [[[-1, 0], [0, 0], [0, 0], [0, 0]],
                [[-1.101601327446325, 0.636009822940780], [0, 0], [0, 0], [0, 0]],
                [[-1.047409162950341, 0.999264438132800], [-1.322675802362066, 0], [0, 0], [0, 0], [0, 0]],
                [[-0.995208762612890, 1.257105737260073], [-1.370067828159648, 0.410249716777560], [0, 0], [0, 0]],
                [[-1.380877331649290, 0.717909590636350], [-0.957676552577410, 1.471124326897576], [-1.502316277745423, 0], [0, 0]],
                [[-1.571490416613821, 0.320896376876750], [-1.381858109025903, 0.971471898746610], [-0.930656530644320, 1.661863282687855], [0, 0]],
                [[-1.612038751019419, 0.589244501373040], [-1.378903203788025, 1.191566766560359], [-0.909867772040510, 1.836451335712793], [-1.684368163384248, 0]],
                [[-1.757408390223183, 0.272867573521960], [-1.636939408646247, 0.822795620374250], [-1.373841209680391, 1.388356567836560], [-0.892869713675890, 1.998325832067540]]]


def helper_calc_sin_and_cos(w0, wx):
    s0 = math.sin(float(w0))
    c0 = math.cos(float(w0))
    sx = math.sin(float(wx))
    cx = math.cos(float(wx))
    return {"s0": s0, "c0": c0, "sx": sx, "cx": cx}


def helper_calc_params(frequency, gain, q):
    A = math.pow(10, gain / 20)
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = res["sx"] * res["sx"] / (wx * q)
    return {"A": A, "alpha": alpha, "c0": res["c0"], "s0": res["s0"], "cx": res["cx"], "sx": res["sx"]}


# RETURN MAGNITUDE IN DB FOR A FREQUENCY f AND A PEQ NUMBER
def get_mag_in_db(filter_type, freq, coeff):
    # for Bell-Sym (0), Bell-ASym (1), Low-Shelving 6dB/oct (3), Low-Shelving 12dB/oct (4), High-Shelving 6dB/oct (5), High-Shelving 12dB/oct (6),
    if 0 == filter_type or 1 == filter_type or 3 == filter_type or 4 == filter_type or 5 == filter_type or 6 == filter_type:
        invert = False
    # for Notch (2), All-Pass 90° (7), All-Pass 180° (8), High-Pass 6dB/oct (9), High-Pass 12dB/oct (10), High-Pass VariQ 12dB/oct (11),
    # Low-Pass 6dB/oct (12), Low-Pass 12dB/oct (13), Low-Pass VariQ 12dB/oct (14), Band-Pass (15)
    elif 2 == filter_type or 7 == filter_type or 8 == filter_type or 9 == filter_type or 10 == filter_type or \
         11 == filter_type or 12 == filter_type or 13 == filter_type or 14 == filter_type or 15 == filter_type:
        invert = True
    else:
        print("Unknown filter type. Filter not found.")
        return 0
    a_vector = []
    a_vector[0] = coeff[0]
    a_vector[1] = coeff[1]
    a_vector[2] = coeff[2]
    b_vector = []
    b_vector[0] = coeff[3]
    b_vector[1] = coeff[4]
    b_vector[2] = coeff[5]

    res_zeros = eval_coefficients(a_vector, freq)
    res_poles = eval_coefficients(b_vector, freq)
    res = res_zeros / res_poles
    temp_mag = abs(res)
    temp_mag = 20 * math.log(10, temp_mag)
    if invert:
        temp_mag = -temp_mag
    return temp_mag


# RETURN PHASE IN DEGREE FOR A FREQUENCY f AND A PEQ NUMBER num
def get_pha_in_degree(filter_type, freq, coeff):
    # for Bell-Sym (0), Bell-ASym (1), Low-Shelving 6dB/oct (3), Low-Shelving 12dB/oct (4), High-Shelving 6dB/oct (5), High-Shelving 12dB/oct (6),
    if 0 == filter_type or 1 == filter_type or 3 == filter_type or 4 == filter_type or 5 == filter_type or 6 == filter_type:
        invert = False
    # for Notch (2), All-Pass 90° (7), All-Pass 180° (8), High-Pass 6dB/oct (9), High-Pass 12dB/oct (10), High-Pass VariQ 12dB/oct (11),
    # Low-Pass 6dB/oct (12), Low-Pass 12dB/oct (13), Low-Pass VariQ 12dB/oct (14), Band-Pass (15)
    elif 2 == filter_type or 7 == filter_type or 8 == filter_type or 9 == filter_type or 10 == filter_type or \
            11 == filter_type or 12 == filter_type or 13 == filter_type or 14 == filter_type or 15 == filter_type:
        invert = True
    else:
        print("Unknown filter type. Filter not found.")
        return 0
    a_vector = []
    a_vector[0] = coeff[0]
    a_vector[1] = coeff[1]
    a_vector[2] = coeff[2]
    b_vector = []
    b_vector[0] = coeff[3]
    b_vector[1] = coeff[4]
    b_vector[2] = coeff[5]

    res_zeros = eval_coefficients(a_vector, freq)
    res_poles = eval_coefficients(b_vector, freq)
    res = res_zeros / res_poles
    temp_pha = cmath.phase(res)
    temp_pha = temp_pha * (180 / math.pi)
    if invert:
        temp_pha = -temp_pha
    return temp_pha


# COMPUTE PEQ COEFFICIENTS (A0, A1, A2, B0, B1, B2) 
def compute_peq_coeff(filter_type, frequency, gain, q):
    if 0 == filter_type:  # Bell-Sym
        a0, a1, a2, b0, b1, b2 = bell_sym_calc_coefficients(frequency, gain, q)
    elif 1 == filter_type:  # Bell-ASym
        a0, a1, a2, b0, b1, b2 = bell_asym_calc_coefficients(frequency, gain, q)
    elif 2 == filter_type:  # Notch
        a0, a1, a2, b0, b1, b2 = notch_calc_coefficients(frequency, q)
    elif 3 == filter_type:  # Low-Shelving 6dB/oct
        a0, a1, a2, b0, b1, b2 = low_shelf_1_calc_coefficients(frequency, gain)
    elif 4 == filter_type:  # Low-Shelving 12dB/oct
        a0, a1, a2, b0, b1, b2 = low_shelf_2_calc_coefficients(frequency, gain)
    elif 5 == filter_type:  # High-Shelving 6dB/oct
        a0, a1, a2, b0, b1, b2 = high_shelf_1_calc_coefficients(frequency, gain)
    elif 6 == filter_type:  # High-Shelving 12dB/oct
        a0, a1, a2, b0, b1, b2 = high_shelf_2_calc_coefficients(frequency, gain)
    elif 7 == filter_type:  # All-Pass 90°
        a0, a1, a2, b0, b1, b2 = all_pass_90_calc_coefficients(frequency)
    elif 8 == filter_type:  # All-Pass 180°
        a0, a1, a2, b0, b1, b2 = all_pass_180_calc_coefficients(frequency, q)
    elif 9 == filter_type:  # High-Pass 6dB/oct
        a0, a1, a2, b0, b1, b2 = highpass_1_calc_coefficients(frequency)
    elif 10 == filter_type:  # High-Pass 12dB/oct
        a0, a1, a2, b0, b1, b2 = highpass_2_calc_coefficients(frequency)
    elif 11 == filter_type:  # High-Pass VariQ 12dB/oct
        a0, a1, a2, b0, b1, b2 = highpass_variq_calc_coefficients(frequency, q)
    elif 12 == filter_type:  # Low-Pass 6dB/oct
        a0, a1, a2, b0, b1, b2 = lowpass_1_calc_coefficients(frequency)
    elif 13 == filter_type:  # Low-Pass 12dB/oct
        a0, a1, a2, b0, b1, b2 = lowpass_2_calc_coefficients(frequency)
    elif 14 == filter_type:  # Low-Pass VariQ 12dB/oct
        a0, a1, a2, b0, b1, b2 = lowpass_variq_calc_coefficients(frequency, q)
    elif 15 == filter_type:  # Band-Pass
        a0, a1, a2, b0, b1, b2 = bandpass_calc_coefficients(frequency, q)
    else:
        print("Unknown filter type. Filter not found.")
        return [0, 0, 0, 0, 0, 0]
    return a0, a1, a2, b0, b1, b2


# COMPUTE COMPLEX VALUE FOR SUM OF ZEROS AND SUM OF POLES IN PEQ COMPUTATION OPERATION
def eval_coefficients(coefs, freq):
    fx = freq/fs
    coefs_len = len(coefs)
    res = 0
    for i in range(0, coefs_len):
        res += coefs[i] * math.exp(complex(0, -i * fx * 2 * math.pi))
    return res


# 0 Bell Symmetric
def bell_sym_calc_coefficients(frequency, gain, q):
    res = helper_calc_params(frequency, gain, q)

    if res["A"] >= 1:
        beta = res["A"] * res["alpha"]
        k = -2 * res["c0"]

        return 1 + res["alpha"], k, 1 - res["alpha"], 1 + beta, k, 1 - beta
    else:
        beta = res["alpha"] / res["A"]
        k = -2 * res["c0"]

        return 1 + beta, k, 1 - beta, 1 + res["alpha"], k, 1 - res["alpha"]


# 1 Bell Asymmetric
def bell_asym_calc_coefficients(frequency, gain, q):
    res = helper_calc_params(frequency, gain, q)
    k = -2 * res["c0"]
    g = res["A"] * res["alpha"]

    return 1 + res["alpha"], k, 1 - res["alpha"], 1 + g, k, 1 - g


# 2 Notch
def notch_calc_coefficients(frequency, q):
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = res["sx"] * res["sx"] / (wx * q)
    k = -2 * res["c0"]

    return 1 + alpha, k, 1 - alpha, 1, k, 1


# 3 Low-Shelving 6dB/oct
def low_shelf_1_calc_coefficients(frequency, gain):
    # Standard first order
    A = math.pow(10, gain / 20)
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)

    if A >= 1:
        g = A * res["sx"]
        return res["sx"] + res["cx"], res["sx"] - res["cx"], 0, g + res["cx"], g - res["cx"], 0
    else:
        A = 1 / A
        return A * res["sx"] + res["cx"], A * res["sx"] - res["cx"], 0, res["sx"] + res["cx"], res["sx"] - res["cx"], 0


# 4 Low-Shelving 12dB/oct
def low_shelf_2_calc_coefficients(frequency, gain):
    # Standard biquad, both numerator and denominator 2nd order
    A = math.pow(10, gain / 20)
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    c1 = 2 * res["sx"] * res["sx"]
    alpha = c1 / (math.sqrt(2) * wx)

    if A >= 1:
        k = (c1 * (A - 1) / 2) + 1
        g = math.sqrt(A) * alpha
        return 1 + alpha, -2 * res["c0"], 1 - alpha, k + g, c1 * (A + 1) - 2, k - g
    else:
        A = 1 / A
        k = (c1 * (A - 1) / 2) + 1
        g = math.sqrt(A) * alpha
        return k + g, c1 * (A + 1) - 2, k - g, 1 + alpha, -2 * res["c0"], 1 - alpha


# 5  High-Shelving 6dB/oct
def high_shelf_1_calc_coefficients(frequency, gain):
    # Standard first order
    A = math.pow(10, gain / 20)
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)

    if A >= 1:
        return res["sx"] + res["cx"], res["sx"] - res["cx"], 0, res["sx"] + A * res["cx"], res["sx"] - A * res["cx"], 0
    else:
        A = 1 / A
        return res["sx"] + A * res["cx"], res["sx"] - A * res["cx"], 0, res["sx"] + res["cx"], res["sx"] - res["cx"],  0


# 6  High-Shelving 12dB/oct
def high_shelf_2_calc_coefficients(frequency, gain):
    # Standard biquad, both numerator and denominator 2nd order
    A = math.pow(10, gain / 20)
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = 2 * res["sx"] * res["sx"] / (math.sqrt(2) * wx)
    cx = 1 + res["c0"]

    if A >= 1:
        k = (cx * (A - 1) / 2) + 1
        g = math.sqrt(A) * alpha
        return 1 + alpha, -2 * res["c0"], 1 - alpha, k + g, 2 - cx * (A + 1), k - g
    else:
        A = 1 / A
        k = cx * (A - 1) / 2 + 1
        g = math.sqrt(A) * alpha
        return k + g, 2 - cx * (A + 1), k - g, 1 + alpha, - 2 * res["c0"], 1 - alpha


# 7 All-Pass 90
def all_pass_90_calc_coefficients(frequency):
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    a = -res["c0"] / (1 + res["s0"])

    return 1, a, 0, a, 1, 0


# 8 All-Pass 180
def all_pass_180_calc_coefficients(frequency, q):
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = res["sx"] * res["sx"] / (wx * q)

    return 1 + alpha, -2 * res["c0"], 1 - alpha, 1 - alpha, -2 * res["c0"], 1 + alpha


# 9 High-Pass 6dB/oct
def highpass_1_calc_coefficients(frequency):
    wx = math.pi * (frequency / fs)
    cx = math.cos(float(wx))
    sx = math.sin(float(wx))

    return sx + cx, sx - cx, 0, cx, -cx, 0


# 10 High-Pass 12dB/oct
def highpass_2_calc_coefficients(frequency):
    q = math.sqrt(2)
    w0 = 2 * math.pi * (frequency / fs)
    c0 = math.cos(float(w0))
    alpha = math.sin(float(w0)) / q
    c1 = 1 + c0

    return 1 + alpha, -2 * c0, 1 - alpha, c1 / 2, -c1, c1 / 2


# 11 High-Pass VariQ 12dB/oct
def highpass_variq_calc_coefficients(frequency, q):
    w0 = 2 * math.pi * (frequency / fs)
    c0 = math.cos(float(w0))
    alpha = math.sin(float(w0)) / (2 * q)
    c1 = 1 + c0

    return 1 + alpha, -2 * c0, 1 - alpha, c1 / 2, -c1, c1 / 2


# 12 Low-Pass 6dB/oct
def lowpass_1_calc_coefficients(frequency):
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)

    return res["sx"] + res["cx"], res["sx"] - res["cx"], 0, res["sx"], res["sx"], 0


# 13 Low-Pass 12dB/oct
def lowpass_2_calc_coefficients(frequency):
    q = math.sqrt(2)
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = math.sin(float(w0)) / q
    c1 = 2 * res["sx"] * res["sx"]

    return 1 + alpha, -2 * res["c0"], 1 - alpha, c1 / 2, c1, c1 / 2


# 14 Low-Pass VariQ 12dB/oct
def lowpass_variq_calc_coefficients(frequency, q):
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = math.sin(float(w0)) / (2 * q)
    c1 = 2 * res["sx"] * res["sx"]

    return 1 + alpha, -2 * res["c0"], 1 - alpha, c1 / 2, c1, c1 / 2


# 15 Band-Pass 6dB/oct
def bandpass_calc_coefficients(frequency, q):
    wx = math.pi * (frequency / fs)
    w0 = 2 * wx
    res = helper_calc_sin_and_cos(w0, wx)
    alpha = res["sx"] * res["sx"] / (wx * q)

    return 1 + alpha, -2 * res["c0"], 1 - alpha, alpha, 0, -alpha


# XOver Available
""""
19 XOVERS AVAILABLE
    "Butterworth 6dB/oct",
    "Butterworth 12dB/oct",
    "Butterworth 18dB/oct",
    "Butterworth 24dB/oct",
    "Butterworth 30dB/oct",
    "Butterworth 36dB/oct",
    "Butterworth 42dB/oct",
    "Butterworth 48dB/oct",
    "Linkwitz-Riley 12dB/oct",
    "Linkwitz-Riley 24dB/oct",
    "Linkwitz-Riley 36dB/oct",
    "Linkwitz-Riley 48dB/oct",
    "Bessel 12dB/oct",
    "Bessel 18dB/oct",
    "Bessel 24dB/oct",
    "Bessel 30dB/oct",
    "Bessel 36dB/oct",
    "Bessel 42dB/oct",
    "Bessel 48dB/oct",
"""


# Compute XOver a and b of coefficient
# type = 0 -> highpass, type = 1 -> lowpass
def compute_xover_coefficients(index, filter_type, freq):
    if index < 12:
        coeff = butterworth_linkwitz_calc_coefficients(index, filter_type, freq)
    else:
        coeff = bessel_calc_coefficients(index - 10, filter_type, freq)
    while len(coeff) < 4:
        coeff.append(ipc_constants.DEFAULT_FILTER)
    return coeff


def bessel_calc_coefficients(index, filter_type, freq):
    high_pass_eq_param = []
    low_pass_eq_param = []
    n = index - 1

    wx = math.pi * (freq / fs)
    w0 = 2 * wx

    for i in range(0, 4):
        a = bessel_coeff[n][i][0]
        b = bessel_coeff[n][i][1]

        temp_filter_coeff = {"a0": 1.0, "a1": 0, "a2": 0, "b0": 1.0, "b1": 0, "b2": 0}

        if filter_type == 0:
            if a != 0:
                if b == 0:
                    sx = math.sin(wx)
                    cx = math.cos(wx)
                    ax = -a * cx
                    a0 = ax + sx

                    temp_filter_coeff["b0"] = ax / a0
                    temp_filter_coeff["b1"] = -temp_filter_coeff["b0"]
                    temp_filter_coeff["a1"] = (sx - ax) / a0
                else:
                    p = a * a + b * b
                    c0 = math.cos(w0)
                    s0 = math.sin(w0)
                    cx = 1 + c0
                    a0 = cx * (p - 1) + 2 * (1 - a * s0)

                    temp_filter_coeff["b0"] = p * cx / a0
                    temp_filter_coeff["b1"] = -2 * temp_filter_coeff["b0"]
                    temp_filter_coeff["b2"] = temp_filter_coeff["b0"]
                    temp_filter_coeff["a1"] = 2 * (2 - cx * (p + 1)) / a0
                    temp_filter_coeff["a2"] = (cx * (p - 1) + 2 * (1 + a * s0)) / a0

                high_pass_eq_param.append(temp_filter_coeff)
        else:
            if a != 0:
                if b == 0:
                    sx = math.sin(wx)
                    cx = math.cos(wx)
                    ax = -a * sx
                    a0 = ax + cx

                    temp_filter_coeff["b0"] = ax / a0
                    temp_filter_coeff["b1"] = temp_filter_coeff["b0"]
                    temp_filter_coeff["a1"] = (ax - cx) / a0
                else:
                    p = a * a + b * b
                    sx = math.sin(wx)
                    c0 = 2 * sx * sx
                    s0 = math.sin(w0)
                    a0 = c0 * (p - 1) + 2 * (1 - a * s0)

                    temp_filter_coeff["b0"] = p * c0 / a0
                    temp_filter_coeff["b1"] = 2 * temp_filter_coeff["b0"]
                    temp_filter_coeff["b2"] = temp_filter_coeff["b0"]
                    temp_filter_coeff["a1"] = 2 * (c0 * (p + 1) - 2) / a0
                    temp_filter_coeff["a2"] = (c0 * (p - 1) + 2 * (1 + a * s0)) / a0

                low_pass_eq_param.append(temp_filter_coeff)

    if 0 == filter_type:
        return high_pass_eq_param
    else:
        return low_pass_eq_param


# XOVER Filter  0-11: Butterworth 6dB/oct to Linkwitz-Riley 48dB/oct
# Compute a butterworth filter up to 8 th order  and  put the coefficients
# in the supplied array. The coefficient array must be at least length 4.
def butterworth_linkwitz_calc_coefficients(index, filter_type, freq):
    high_pass_eq_param = []
    low_pass_eq_param = []
    order = 0
    linkwitz = 0

    if index > 7:
        linkwitz = 1

#    if index == 2 or index == 3 or index == 4 or index == 5 or index == 6 or index == 7 or index == 10 or index == 11:
#        if filter_type == 0:
#            freq = freq / 2
#        if filter_type == 1:
#            freq = freq * 2

    if index == 0:
        order = 1
    elif index == 1:
        order = 2
    elif index == 2:
        order = 3
    elif index == 3:
        order = 4
    elif index == 4:
        order = 5
    elif index == 5:
        order = 6
    elif index == 6:
        order = 7
    elif index == 7:
        order = 8
    elif index == 8:
        order = 2
    elif index == 9:
        order = 4
    elif index == 10:
        order = 6
    elif index == 11:
        order = 8

    wx = math.pi * (freq / fs)
    sx = math.sin(wx)
    w0 = 2 * wx
    s0 = math.sin(w0)
    c0 = math.cos(w0)
    sx2 = sx * sx

    n = order

    if linkwitz == 1:
        # Constructed from 2 half - order Butterworths
        n = n / 2

    # number of biquad sections
    nk = n / 2

    for k in range(1, math.floor(nk) + 1):
        filter_coeff = {"a0": 1.0, "a1": 0, "a2": 0, "b0": 1.0, "b1": 0, "b2": 0}

        alpha = s0 * math.cos(math.pi * (n - 2 * k + 1) / (2 * n))
        a0 = 1 + alpha

        if filter_type == 0:
            A = (1 + c0) / a0
            filter_coeff["b1"] = -A
            A /= 2
        else:
            A = sx2 / a0
            filter_coeff["b1"] = 2 * A

        filter_coeff["b0"] = A
        filter_coeff["b2"] = A
        filter_coeff["a1"] = -2 * c0 / a0
        filter_coeff["a2"] = (1 - alpha) / a0

        if filter_type == 0:
            high_pass_eq_param.append(filter_coeff)
            if linkwitz == 1:
                high_pass_eq_param.append(filter_coeff)
        else:
            low_pass_eq_param.append(filter_coeff)
            if linkwitz == 1:
                low_pass_eq_param.append(filter_coeff)

    # Add the necessary sections to complete odd order filters
    if n % 2 != 0:
        if linkwitz == 1:
            if filter_type == 0:
                qEq = 0.5

                w0 = 2 * math.pi * (freq / fs)
                c0 = math.cos(w0)
                alpha = math.sin(w0) / (2 * qEq)
                c1 = 1 + c0

                filter_coeff_linkwitz = {"a0": 1 + alpha, "a1": -2 * c0, "a2": 1 - alpha, "b0": c1 / 2, "b1": -c1, "b2": c1 / 2}
                high_pass_eq_param.append(filter_coeff_linkwitz)
            else:
                qEq = 0.5

                wx = math.pi * (freq / fs)
                w0 = 2 * wx
                c0 = math.cos(w0)
                alpha = math.sin(w0) / (2 * qEq)
                sx = math.sin(wx)
                c1 = 2 * sx * sx

                filter_coeff_linkwitz = {"a0": 1 + alpha, "a1": -2 * c0, "a2": 1 - alpha, "b0": c1 / 2, "b1": c1, "b2": c1 / 2}
                low_pass_eq_param.append(filter_coeff_linkwitz)
        else:
            if filter_type == 0:
                wx = math.pi * (freq / fs)
                cx = math.cos(wx)
                sx = math.sin(wx)

                temp_filter_coeff = {"a0": sx + cx, "a1": sx - cx, "a2": 0, "b0": cx, "b1": -cx, "b2": 0}
                high_pass_eq_param.append(temp_filter_coeff)
            else:
                # Standard first order
                wx = math.pi * (freq / fs)
                cx = math.cos(wx)
                sx = math.sin(wx)

                temp_filter_coeff = {"a0": sx + cx, "a1": sx - cx, "a2": 0, "b0": sx, "b1": sx, "b2": 0}
                low_pass_eq_param.append(temp_filter_coeff)

    if 0 == filter_type:
        return high_pass_eq_param
    else:
        return low_pass_eq_param
