import base64
import binascii
import dbus
import json
import sys
import os
import zipfile

import ipc_common
import ipc_util


sw_update_in_progress = False


def get_ip_address_of_neighbour_device(index: int):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.list_neighbours())
        neighbours = json.loads(data)
        return neighbours[index]['ip_addr']
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    except (json.JSONDecodeError, KeyError, IndexError):
        return "<NONE>"


def protocol_test(timestamp):
    # This command is purely for testing the chain of execution and does nothing else.
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.protocol_test())
        if data is None:
            return create_response_error('protocol_test',
                                         'ERROR: Obtaining testing the procotol.',
                                         timestamp)
        else:
            return create_response_ok('protocol_test', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def eth_config(config, timestamp):
    try:
        if config is None:
            proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
            if not proxy_object.eth_dhcp_config():
                return create_response_error('eth_dhcp', 'ERROR: Ethernet DHCP configuration failed.', timestamp)
            else:
                return create_response_ok('eth_dhcp', timestamp=timestamp)
        elif config == '':
            return create_response_error('eth_static', 'ERROR: Ethernet static configuration failed.', timestamp)
        else:
            proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
            if not proxy_object.eth_static_config(config):
                return create_response_error('eth_static', 'ERROR: Ethernet static configuration failed.', timestamp)
            else:
                return create_response_ok('eth_static', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def readback_ctl(enable, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if enable:
            cmd_id = 'start_readback'
            res: bool = proxy_object.start_peak_readback()
        else:
            cmd_id = 'stop_readback'
            res: bool = proxy_object.stop_peak_readback()

        if not res:
            return create_response_error(cmd_id, 'ERROR: Peak readback enable/disable failed.', timestamp)
        else:
            return create_response_ok(cmd_id, timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def readback_get(timestamp):
    # To save precious milliseconds, skip D-Bus and read directly from ipc_common.PEAK_READBACK_FILE
    try:
        with open(ipc_common.PEAK_READBACK_FILE, "r", os.O_NONBLOCK) as readback_file:
            resp = create_response_ok('get_readback', readback_file.read(), timestamp)
    except OSError:
        resp = create_response_error('get_readback', 'ERROR: Cannot open readback file.', timestamp)
    return resp

# RETRIEVE SESSION FILE
def session_get(timestamp):
    # To save precious milliseconds, skip D-Bus and read directly from ipc_common.PEAK_READBACK_FILE
    try:
        with open(ipc_common.SESSION_FILE, "r", os.O_NONBLOCK) as session_file:
            resp = create_response_ok('get_session', json.load(session_file), timestamp)
    except OSError:
        resp = create_response_error('get_session', 'ERROR: Cannot open session file.', timestamp)
    return resp

# SET SESSION FILE
def session_set(dict, timestamp):
    # To save precious milliseconds, skip D-Bus and read directly from ipc_common.PEAK_READBACK_FILE
    try:
        json_object = json.dumps(dict, indent=4)
        with open(ipc_common.SESSION_FILE, "w", os.O_NONBLOCK) as session_file:
            session_file.write(json_object)
            os.system('sync')
            resp = create_response_ok('set_session', timestamp)
    except OSError:
        resp = create_response_error('set_session', 'ERROR: Cannot write session file.', timestamp)
    return resp


def software_update(b64payload: str, timestamp):
    global sw_update_in_progress

    if sw_update_in_progress:
        return create_response_error('software_update', 'ERROR: Update is already in progress.', timestamp)

    sw_update_in_progress = True

    # Remove previous update file, if present
    try:
        os.remove(ipc_common.SW_UPDATE_FILE)
    except (FileNotFoundError, PermissionError):
        pass

    # Clean and recreate update folder (where update will be unpacked)
    os.system("/bin/rm -rf " + ipc_common.SW_UPDATE_FOLDER)
    os.mkdir(ipc_common.SW_UPDATE_FOLDER)

    try:
        # Decode downloaded base64 string to binary file
        binary_data = base64.b64decode(b64payload)
        binfile = open(ipc_common.SW_UPDATE_FILE, 'wb')
        binfile.write(binary_data)
        binfile.flush()
        binfile.close()

        # Test received archive for corruption and unpack on success
        with zipfile.ZipFile(ipc_common.SW_UPDATE_FILE, 'r') as zf:
            zf.testzip()
            zf.extractall(ipc_common.SW_UPDATE_FOLDER)

        if os.system('chmod +x ' + ipc_common.SW_UPDATE_RUNNER) != 0:
            resp = create_response_error('software_update', 'ERROR: Could not find update runner.', timestamp)
        elif os.system('nohup ' + ipc_common.SW_UPDATE_RUNNER + ' &') != 0:
            resp = create_response_error('software_update', 'ERROR: Unable to start update runner.', timestamp)
        else:
            resp = create_response_ok('software_update', 'Update launched.', timestamp)
    except binascii.Error:
        resp = create_response_error('software_update', 'ERROR: Unable to decode the payload.', timestamp)
    except (zipfile.BadZipFile, ValueError, RuntimeError):
        resp = create_response_error('software_update', 'ERROR: The archive is damaged.', timestamp)
    except OSError:
        resp = create_response_error('software_update', 'ERROR: Unable to create temporary file.', timestamp)

    sw_update_in_progress = False
    return resp


def hotspot_enable(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.hotspot_enabled(enabled):
            return create_response_error('hotspot_enable', 'ERROR: Hotspot enabled/disabled failed.', timestamp)
        else:
            return create_response_ok('hotspot_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_hotspot_config(json_config, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.hotspot_config_set(json_config):
            return create_response_error('set_hotspot_config', 'ERROR: Hotspot config failed.', timestamp)
        else:
            return create_response_ok('set_hotspot_config', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def cloud_enable(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.cloud_enabled(enabled):
            return create_response_error('cloud_enable', 'ERROR: Cloud enabled/disabled failed.', timestamp)
        else:
            return create_response_ok('cloud_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def global_preset_set(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.global_preset_file_set(path):
            return create_response_error('global_preset_set', 'ERROR: Global preset set failed.', timestamp)
        else:
            return create_response_ok('global_preset_set', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_set(channel, path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_preset_file_set(channel, path):
            return create_response_error('speaker_preset_set', 'ERROR: Speaker preset set failed.', timestamp)
        else:
            return create_response_ok('speaker_preset_set', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_home_page_status_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_home_page_status_get())
        if data is None:
            return create_response_error('get_data_home_page_status',
                                         'ERROR: Obtaining data for home page status failed.',
                                         timestamp)
        else:
            return create_response_ok('get_data_home_page_status', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_audio_delay_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_audio_delay_get())
        if data is None:
            return create_response_error('get_data_audio_delay',
                                         'ERROR: Obtaining data for audio delay failed.',
                                         timestamp)
        else:
            return create_response_ok('get_data_audio_delay', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_audio_gain_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_audio_gain_get())
        if data is None:
            return create_response_error('get_data_audio_gain',
                                         'ERROR: Obtaining data for audio gain failed.',
                                         timestamp)
        else:
            return create_response_ok('get_data_audio_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_audio_source_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_audio_source_get())
        if data is None:
            return create_response_error('get_data_audio_source',
                                         'ERROR: Obtaining data for audio source failed.',
                                         timestamp)
        else:
            return create_response_ok('get_data_audio_source', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_hotspot_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_hotspot_get())
        if data is None:
            return create_response_error('get_data_hotspot', 'ERROR: data obtaining for hotspot failed.', timestamp)
        else:
            return create_response_ok('get_data_hotspot', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_cloud_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_cloud_get())
        if data is None:
            return create_response_error('get_data_cloud', 'ERROR: data obtaining for cloud failed.', timestamp)
        else:
            return create_response_ok('get_data_cloud', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_network_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_network_get())
        if data is None:
            return create_response_error('get_data_network', 'ERROR: data obtaining for network failed.', timestamp)
        else:
            return create_response_ok('get_data_network', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def data_about_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.data_about_get())
        if data is None:
            return create_response_error('get_data_about', 'ERROR: data obtaining for about failed.', timestamp)
        else:
            return create_response_ok('get_data_about', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def group_mute_get(group_number, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_mute_get(group_number))
        if data is None:
            return create_response_error('get_group_mute', 'ERROR: data obtaining for group mute failed.', timestamp)
        else:
            return create_response_ok('get_group_mute', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_gain_get(group_number, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_gain_get(group_number))
        if data is None:
            return create_response_error('get_group_gain',
                                         'ERROR: data obtaining for group gain failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_polarity_get(group_number, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_polarity_get(group_number))
        if data is None:
            return create_response_error('get_group_polarity',
                                         'ERROR: data obtaining for group polarity failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_polarity', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_delay_enable_get(group_number, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_delay_enable_get(group_number))
        if data is None:
            return create_response_ok('get_group_delay_enable',
                                      'ERROR: data obtaining for group delay enable failed.',
                                      timestamp)
        else:
            return create_response_ok('get_group_delay_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_delay_value_get(group_number, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_delay_value_get(group_number))
        if data is None:
            return create_response_error('get_group_delay_value',
                                         'ERROR: data obtaining for group delay failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_delay_value', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_name_get(group_number, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_name_get(group_number))
        if data is None:
            return create_response_error('get_group_name', 'ERROR: data obtaining for group name failed.', timestamp)
        else:
            return create_response_ok('get_group_name', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_mute_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_mute_get(channel))
        if data is None:
            return create_response_error('get_solo_mute', 'ERROR: data obtaining for solo mute failed.', timestamp)
        else:
            return create_response_ok('get_solo_mute', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_gain_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_gain_get(channel))
        if data is None:
            return create_response_error('get_solo_gain', 'ERROR: data obtaining for solo gain failed.', timestamp)
        else:
            return create_response_ok('get_solo_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_polarity_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_polarity_get(channel))
        if data is None:
            return create_response_error('get_solo_polarity',
                                         'ERROR: data obtaining for solo polarity failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_polarity', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_delay_enable_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_delay_enable_get(channel))
        if data is None:
            return create_response_error('get_solo_delay_enable',
                                         'ERROR: data obtaining for solo delay enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_delay_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_delay_value_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_delay_value_get(channel))
        if data is None:
            return create_response_error('get_solo_delay_value',
                                         'ERROR: data obtaining for solo delay failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_delay_value', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_hp_enable_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_xover_hp_enable_get(channel))
        if data is None:
            return create_response_error('get_solo_xover_hp_enable',
                                         'ERROR: data obtaining for solo xover high pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_xover_hp_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_hp_type_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_xover_hp_type_get(channel))
        if data is None:
            return create_response_error('get_solo_xover_hp_type',
                                         'ERROR: data obtaining for solo xover high pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_xover_hp_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_hp_freq_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_xover_hp_freq_get(channel))
        if data is None:
            return create_response_error('get_solo_xover_hp_freq',
                                         'ERROR: data obtaining for solo xover high pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_xover_hp_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_lp_enable_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_xover_lp_enable_get(channel))
        if data is None:
            return create_response_error('get_solo_xover_lp_enable',
                                         'ERROR: data obtaining for solo xover low pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_xover_lp_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_lp_type_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_xover_lp_type_get(channel))
        if data is None:
            return create_response_error('get_solo_xover_lp_type',
                                         'ERROR: data obtaining for solo xover low pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_xover_lp_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_lp_freq_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_xover_lp_freq_get(channel))
        if data is None:
            return create_response_error('get_solo_xover_lp_freq',
                                         'ERROR: data obtaining for solo xover low pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_xover_lp_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_rms_thres_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_lim_rms_thres_get(channel))
        if data is None:
            return create_response_error('get_solo_lim_rms_thres',
                                         'ERROR: data obtaining for solo lim rms threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_lim_rms_thres', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_rms_attack_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_lim_rms_attack_get(channel))
        if data is None:
            return create_response_error('get_solo_lim_rms_attack',
                                         'ERROR: data obtaining for solo lim rms attack failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_lim_rms_attack', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_rms_release_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_lim_rms_release_get(channel))
        if data is None:
            return create_response_error('get_solo_lim_rms_release',
                                         'ERROR: data obtaining for solo lim rms release failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_lim_rms_release', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_peak_thres_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_lim_peak_thres_get(channel))
        if data is None:
            return create_response_error('get_solo_lim_peak_thres',
                                         'ERROR: data obtaining for solo lim peak threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_lim_peak_thres', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_peak_release_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_lim_peak_release_get(channel))
        if data is None:
            return create_response_error('get_solo_lim_peak_release',
                                         'ERROR: data obtaining for solo lim peak release failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_lim_peak_release', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_channel_source_pri_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_channel_source_pri_get(channel))
        if data is None:
            return create_response_error('get_solo_channel_source_pri',
                                         'ERROR: data obtaining for solo channel primary source failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_channel_source_pri', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_channel_source_sec_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_channel_source_sec_get(channel))
        if data is None:
            return create_response_error('get_solo_channel_source_sec',
                                         'ERROR: data obtaining for solo channel secondary source failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_channel_source_sec', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def solo_channel_source_mixer_get(channel, source, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_channel_source_mixer_get(channel, source))
        if data is None:
            return create_response_error('get_solo_channel_source_mixer',
                                         'ERROR: data obtaining for solo channel primary source failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_channel_source_mixer', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_channel_name_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_channel_name_get(channel))
        if data is None:
            return create_response_error('get_solo_channel_name',
                                         'ERROR: data obtaining for solo channel name failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_channel_name', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_enable_get(group_number, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_peq_enable_get(group_number, peq_channel))
        if data is None:
            return create_response_error('get_group_peq_enable',
                                         'ERROR: data obtaining for group peq enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_peq_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_type_get(group_number, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_peq_type_get(group_number, peq_channel))
        if data is None:
            return create_response_error('get_group_peq_type',
                                         'ERROR: data obtaining for group peq type failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_peq_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_freq_get(group_number, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_peq_freq_get(group_number, peq_channel))
        if data is None:
            return create_response_error('get_group_peq_freq',
                                         'ERROR: data obtaining for group peq frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_peq_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_gain_get(group_number, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_peq_gain_get(group_number, peq_channel))
        if data is None:
            return create_response_error('get_group_peq_gain',
                                         'ERROR: data obtaining for group peq gain failed.',
                                         timestamp)
        else:
            return create_response_ok('get_group_peq_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_q_get(group_number, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.group_peq_q_get(group_number, peq_channel))
        if data is None:
            return create_response_error('get_group_peq_q', 'ERROR: data obtaining for group peq q failed.', timestamp)
        else:
            return create_response_ok('get_group_peq_q', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_mute_set(group_number, muted, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_mute_set(group_number, muted):
            return create_response_error('set_group_mute', 'ERROR: setting group mute failed.', timestamp)
        else:
            return create_response_ok('set_group_mute', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_gain_set(group_number, gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_gain_set(group_number, gain):
            return create_response_error('set_group_gain', 'ERROR: setting group gain failed.', timestamp)
        else:
            return create_response_ok('set_group_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_polarity_set(group_number, polarity, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_polarity_set(group_number, polarity):
            return create_response_error('set_group_polarity', 'ERROR: setting group polarity failed.', timestamp)
        else:
            return create_response_ok('set_group_polarity', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_delay_enable_set(group_number, delay_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_delay_enable_set(group_number, delay_enabled):
            return create_response_error('set_group_delay_enable',
                                         'ERROR: setting group delay enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_group_delay_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_delay_value_set(group_number, delay_value, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_delay_value_set(group_number, delay_value):
            return create_response_error('set_group_delay_value', 'ERROR: setting group delay value failed.', timestamp)
        else:
            return create_response_ok('set_group_delay_value', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_name_set(group_number, name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_name_set(group_number, name):
            return create_response_error('set_group_name', 'ERROR: setting group name failed.', timestamp)
        else:
            return create_response_ok('set_group_name', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_enable_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_peq_enable_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_solo_peq_enable',
                                         'ERROR: data obtaining for solo peq enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_peq_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_type_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_peq_type_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_solo_peq_type',
                                         'ERROR: data obtaining for solo peq type failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_peq_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_freq_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_peq_freq_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_solo_peq_freq',
                                         'ERROR: data obtaining for solo peq frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_peq_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_gain_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_peq_gain_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_solo_peq_gain',
                                         'ERROR: data obtaining for solo peq gain failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_peq_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_q_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.solo_peq_q_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_solo_peq_q',
                                         'ERROR: data obtaining for solo peq q failed.',
                                         timestamp)
        else:
            return create_response_ok('get_solo_peq_q', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def amp_module_disable_set(channel, disabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.amp_module_disable_set(channel, disabled):
            return create_response_error('set_amp_module_disable', 'ERROR: setting amp module disable failed.', timestamp)
        else:
            return create_response_ok('set_amp_module_disable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def amp_stage_disable_set(disabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.amp_stage_disable_set(disabled):
            return create_response_error('set_amp_stage_disable', 'ERROR: setting amp stage disable failed.', timestamp)
        else:
            return create_response_ok('set_amp_stage_disable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    
def solo_mute_set(channel, muted, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_mute_set(channel, muted):
            return create_response_error('set_solo_mute', 'ERROR: setting solo mute failed.', timestamp)
        else:
            return create_response_ok('set_solo_mute', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_gain_set(channel, gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_gain_set(channel, gain):
            return create_response_error('set_solo_gain', 'ERROR: setting solo gain failed.', timestamp)
        else:
            return create_response_ok('set_solo_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_polarity_set(channel, polarity, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_polarity_set(channel, polarity):
            return create_response_error('set_solo_polarity', 'ERROR: setting solo polarity failed.', timestamp)
        else:
            return create_response_ok('set_solo_polarity', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_delay_enable_set(channel, delay_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_delay_enable_set(channel, delay_enabled):
            return create_response_error('set_solo_delay_enable', 'ERROR: setting solo delay enable failed.', timestamp)
        else:
            return create_response_ok('set_solo_delay_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_delay_value_set(channel, delay_value, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_delay_value_set(channel, delay_value):
            return create_response_error('set_solo_delay_value',
                                         'ERROR: setting solo delay value failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_delay_value', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_hp_enable_set(channel, xover_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_xover_hp_enable_set(channel, xover_enabled):
            return create_response_error('set_solo_xover_hp_enable',
                                         'ERROR: setting solo xover high pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_xover_hp_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_hp_type_set(channel, xover_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_xover_hp_type_set(channel, xover_type):
            return create_response_error('set_solo_xover_hp_type',
                                         'ERROR: setting solo xover high pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_xover_hp_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_hp_freq_set(channel, xover_freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_xover_hp_freq_set(channel, xover_freq):
            return create_response_error('set_solo_xover_hp_freq',
                                         'ERROR: setting solo xover high pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_xover_hp_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_lp_enable_set(channel, xover_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_xover_lp_enable_set(channel, xover_enabled):
            return create_response_error('set_solo_xover_lp_enable',
                                         'ERROR: setting solo xover low pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_xover_lp_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_lp_type_set(channel, xover_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_xover_lp_type_set(channel, xover_type):
            return create_response_error('set_solo_xover_lp_type',
                                         'ERROR: setting solo xover low pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_xover_lp_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_xover_lp_freq_set(channel, xover_freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_xover_lp_freq_set(channel, xover_freq):
            return create_response_error('set_solo_xover_lp_freq',
                                         'ERROR: setting solo xover low pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_xover_lp_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_rms_thres_set(channel, thres, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_lim_rms_thres_set(channel, thres):
            return create_response_error('set_solo_lim_rms_thres',
                                         'ERROR: setting solo lim rms threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_lim_rms_thres', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_rms_attack_set(channel, attack, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_lim_rms_attack_set(channel, attack):
            return create_response_error('set_solo_lim_rms_attack',
                                         'ERROR: setting solo lim rms attack failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_lim_rms_attack', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_rms_release_set(channel, release, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_lim_rms_release_set(channel, release):
            return create_response_error('set_solo_lim_rms_release',
                                         'ERROR: setting solo lim rms release failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_lim_rms_release', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_peak_thres_set(channel, thres, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_lim_peak_thres_set(channel, thres):
            return create_response_error('set_solo_lim_peak_thres',
                                         'ERROR: setting solo lim peak threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_lim_peak_thres', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_lim_peak_release_set(channel, release, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_lim_peak_release_set(channel, release):
            return create_response_error('set_solo_lim_peak_release',
                                         'ERROR: setting solo lim peak release failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_lim_peak_release', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_channel_source_pri_set(channel, source, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_channel_source_pri_set(channel, source):
            return create_response_error('set_solo_channel_source_pri',
                                         'ERROR: setting solo channel source primary failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_channel_source_pri', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_channel_source_sec_set(channel, source, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_channel_source_sec_set(channel, source):
            return create_response_error('set_solo_channel_source_sec',
                                         'ERROR: setting solo channel source secondary failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_channel_source_sec', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def solo_channel_source_mixer_set(channel, source, gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_channel_source_mixer_set(channel, source, gain):
            return create_response_error('set_solo_channel_mixer_gain',
                                         'ERROR: setting solo channel source mixer failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_channel_mixer_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def settings_set_freq_gen_frequency(frequency, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.freq_gen_frequency_set(frequency):
            return create_response_error('freq_gen_frequency_set',
                                         'ERROR: changing frequency freq gen set failed.',
                                         timestamp)
        else:
            return create_response_ok('set_freq_gen_frequency', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def settings_set_freq_gen_gain(gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.freq_gen_gain_set(gain):
            return create_response_error('freq_gen_frequency_set',
                                         'ERROR: changing gain freq gen set failed.',
                                         timestamp)
        else:
            return create_response_ok('set_freq_gen_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def settings_set_pink_gen_gain(gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.pink_noise_gain_set(gain):
            return create_response_error('pink_gen_gain_set',
                                         'ERROR: changing pink noise gain set failed.',
                                         timestamp)
        else:
            return create_response_ok('set_pink_gen_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_channel_name_set(channel, name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_channel_name_set(channel, name):
            return create_response_error('set_solo_channel_name',
                                         'ERROR: setting solo channel name failed.',
                                         timestamp)
        else:
            return create_response_ok('set_solo_channel_name', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_enable_set(group_number, peq_channel, enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_peq_enable_set(group_number, peq_channel, enabled):
            return create_response_error('set_group_peq_enable', 'ERROR: setting group peq enable failed.', timestamp)
        else:
            return create_response_ok('set_group_peq_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_type_set(group_number, peq_channel, peq_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_peq_type_set(group_number, peq_channel, peq_type):
            return create_response_error('set_group_peq_type', 'ERROR: setting group peq type failed.', timestamp)
        else:
            return create_response_ok('set_group_peq_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_freq_set(group_number, peq_channel, freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_peq_freq_set(group_number, peq_channel, freq):
            return create_response_error('set_group_peq_freq', 'ERROR: setting group peq frequency failed.', timestamp)
        else:
            return create_response_ok('set_group_peq_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_gain_set(group_number, peq_channel, gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_peq_gain_set(group_number, peq_channel, gain):
            return create_response_error('set_group_peq_gain', 'ERROR: setting group peq gain failed.', timestamp)
        else:
            return create_response_ok('set_group_peq_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def group_peq_q_set(group_number, peq_channel, q, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_peq_q_set(group_number, peq_channel, q):
            return create_response_error('set_group_peq_q', 'ERROR: setting group peq q failed.', timestamp)
        else:
            return create_response_ok('set_group_peq_q', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def group_peq_set(group_number, peq_channel, peq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.group_peq_set(group_number, peq_channel, peq):
            return create_response_error('set_group_peq', 'ERROR: setting group peq failed.', timestamp)
        else:
            return create_response_ok('set_group_peq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def solo_peq_set(channel, peq_channel, peq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_peq_set(channel, peq_channel, peq):
            return create_response_error('set_solo_peq', 'ERROR: setting solo peq failed.', timestamp)
        else:
            return create_response_ok('set_solo_peq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_enable_set(channel, peq_channel, peq_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_peq_enable_set(channel, peq_channel, peq_enabled):
            return create_response_error('set_solo_peq_enable', 'ERROR: setting solo peq enable failed.', timestamp)
        else:
            return create_response_ok('set_solo_peq_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_type_set(channel, peq_channel, peq_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_peq_type_set(channel, peq_channel, peq_type):
            return create_response_error('set_solo_peq_type', 'ERROR: setting solo peq type failed.', timestamp)
        else:
            return create_response_ok('set_solo_peq_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_freq_set(channel, peq_channel, peq_freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_peq_freq_set(channel, peq_channel, peq_freq):
            return create_response_error('set_solo_peq_freq', 'ERROR: setting solo peq frequency failed.', timestamp)
        else:
            return create_response_ok('set_solo_peq_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_gain_set(channel, peq_channel, peq_gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_peq_gain_set(channel, peq_channel, peq_gain):
            return create_response_error('set_solo_peq_gain', 'ERROR: setting solo peq gain failed.', timestamp)
        else:
            return create_response_ok('set_solo_peq_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def solo_peq_q_set(channel, peq_channel, peq_q, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.solo_peq_q_set(channel, peq_channel, peq_q):
            return create_response_error('set_solo_peq_q', 'ERROR: setting solo peq q failed.', timestamp)
        else:
            return create_response_ok('set_solo_peq_q', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def assign_channel_to_group_set(channel, group, assign_value, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.assign_channel_to_group_set(channel, group, assign_value):
            return create_response_error('set_assign_channel_to_group',
                                         'ERROR: setting assign channel to group failed.',
                                         timestamp)
        else:
            return create_response_ok('set_assign_channel_to_group', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def assign_channel_to_group_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.assign_channel_to_group_get(channel))
        if data is None:
            return create_response_error('get_assign_channel_to_group',
                                         'ERROR: data obtaining for assign channel to group failed.',
                                         timestamp)
        else:
            return create_response_ok('get_assign_channel_to_group', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_list_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_preset_list_names_get())
        if data is None:
            return create_response_error('get_speaker_preset_list',
                                         'ERROR: data obtaining for speaker preset list.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_preset_list_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_file_get(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_preset_file_get(path))
        if data is None:
            return create_response_error('get_speaker_preset_file',
                                         'ERROR: data obtaining for speaker preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_preset_file', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def all_speaker_preset_files_from_folder_get(folder, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.all_speaker_preset_files_from_folder_get(folder))
        if data is None:
            return create_response_error('get_all_speaker_preset_files_from_folder',
                                         'ERROR: data obtaining for speaker preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_all_speaker_preset_files_from_folder', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

        
def all_speaker_preset_files_from_folder_set(folder, files, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.all_speaker_preset_files_from_folder_set(folder, files))
        if data is True:
            return create_response_error('set_all_speaker_preset_files_from_folder',
                                         'File already exist in folder',
                                         timestamp)
        else:
            return create_response_ok('set_all_speaker_preset_files_from_folder', "data", timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_files_from_folder_get(folder, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_preset_files_from_folder_get(folder))
        if data is None:
            return create_response_error('get_speaker_preset_files_from_folder',
                                         'ERROR: data obtaining for speaker preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_preset_files_from_folder', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_files_from_folder_set(folder, files, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_preset_files_from_folder_set(folder, files))
        if data is True:
            return create_response_error('set_speaker_preset_files_from_folder',
                                         'File already exist in folder',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_preset_files_from_folder', "data", timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_folder_names_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_preset_folder_names_get())
        if data is None:
            return create_response_error('get_speaker_preset_folder_names',
                                         'ERROR: data obtaining names for speaker preset folder.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_preset_folder_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_preset_file_names_get(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_preset_file_names_get(path))
        if data is None:
            return create_response_error('get_speaker_preset_file_names',
                                         'ERROR: data obtaining names for speaker preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_preset_file_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def all_speaker_preset_list_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.all_speaker_preset_list_names_get())
        if data is None:
            return create_response_error('get_all_speaker_preset_list',
                                         'ERROR: data obtaining for speaker all preset list.',
                                         timestamp)
        else:
            return create_response_ok('get_all_speaker_preset_list_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def all_speaker_preset_file_get(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.all_speaker_preset_file_get(path))
        if data is None:
            return create_response_error('get_all_speaker_preset_file',
                                         'ERROR: data obtaining for speaker all preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_all_speaker_preset_file', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def all_speaker_preset_folder_names_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.all_speaker_preset_folder_names_get())
        if data is None:
            return create_response_error('get_all_speaker_preset_folder_names',
                                         'ERROR: data obtaining names for all speaker preset folder.',
                                         timestamp)
        else:
            return create_response_ok('get_all_speaker_preset_folder_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def all_speaker_preset_file_names_get(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.all_speaker_preset_file_names_get(path))
        if data is None:
            return create_response_error('get_all_speaker_preset_file_names',
                                         'ERROR: data obtaining names for all speaker preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_preset_file_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    

def global_preset_file_get(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.global_preset_file_get(path))
        if data is None:
            return create_response_error('get_global_preset_file',
                                         'ERROR: data obtaining names for global preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_global_preset_file', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def global_preset_folder_names_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.global_preset_folder_names_get())
        if data is None:
            return create_response_error('get_global_preset_folder_names',
                                         'ERROR: data obtaining names for global preset folder.',
                                         timestamp)
        else:
            return create_response_ok('get_global_preset_folder_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def global_preset_file_names_get(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.global_preset_file_names_get(path))
        if data is None:
            return create_response_error('get_global_preset_file_names',
                                         'ERROR: data obtaining names for global preset file.',
                                         timestamp)
        else:
            return create_response_ok('get_global_preset_file_names', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def delete_global_preset_folder(folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.delete_global_preset_folder(folder_name):
            return create_response_error('delete_global_preset_folder',
                                         'ERROR: deleting global preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('delete_global_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def create_global_preset_folder(folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.create_global_preset_folder(folder_name):
            return create_response_error('create_global_preset_folder',
                                         'ERROR: creating global preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('create_global_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_global_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.global_preset_file_set(path):
            return create_response_error('set_global_preset_file',
                                         'ERROR: setting global preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('set_global_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def delete_global_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.delete_global_preset_file(path):
            return create_response_error('delete_global_preset_file',
                                         'ERROR: deleting global preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('delete_global_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def delete_speaker_preset_folder(folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.delete_speaker_preset_folder(folder_name):
            return create_response_error('delete_speaker_preset_folder',
                                         'ERROR: deleting speaker preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('delete_speaker_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def create_speaker_preset_folder(folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.create_speaker_preset_folder(folder_name):
            return create_response_error('create_speaker_preset_folder',
                                         'ERROR: creating speaker preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('create_speaker_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def create_speaker_preset_file(path, channel, info, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.create_speaker_preset_file(path, channel, info):
            return create_response_error('create_speaker_preset_file',
                                         'ERROR: uploader speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('create_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERRROR: IPC Service unreachable: {e.get_dbus_name()}'


def upload_speaker_preset_file(path, preset, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.upload_speaker_preset_file(path, preset):
            return create_response_error('upload_speaker_preset_file',
                                         'ERROR: uploader speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('upload_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERRROR: IPC Service unreachable: {e.get_dbus_name()}'

def edit_speaker_preset_file(path, channel, info, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.edit_speaker_preset_file(path, channel, info):
            return create_response_error('edit_speaker_preset_file',
                                         'ERROR: editing speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('edit_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_speaker_preset_file(path, channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_preset_file_set(path, channel):
            return create_response_error('set_speaker_preset_file',
                                         'ERROR: setting speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    
def set_label_speaker_preset_file(path, channel, label, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_preset_file_label_set(path, channel, label):
            return create_response_error('set_label_speaker_preset_file',
                                         'ERROR: setting speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('set_label_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_all_speaker_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.all_speaker_preset_file_set(path):
            return create_response_error('set_all_speaker_preset_file',
                                         'ERROR: setting speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('set_all_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def rename_speaker_preset_file(path, new_file_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.rename_speaker_preset_file(path, new_file_name):
            return create_response_error('rename_speaker_preset_file',
                                         'ERROR: renaming speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('rename_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def rename_speaker_preset_folder(folder_name, new_folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.rename_speaker_preset_folder(folder_name, new_folder_name):
            return create_response_error('rename_speaker_preset_folder',
                                         'ERROR: renaming speaker preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('rename_speaker_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def delete_all_speaker_preset_folder(folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.delete_all_speaker_preset_folder(folder_name):
            return create_response_error('delete_all_speaker_preset_folder',
                                         'ERROR: deleting all speaker preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('delete_all_speaker_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def create_all_speaker_preset_folder(folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.create_all_speaker_preset_folder(folder_name):
            return create_response_error('create_all_speaker_preset_folder',
                                         'ERROR: creating all speaker preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('create_all_speaker_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def create_all_speaker_preset_file(path, info, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.create_all_speaker_preset_file(path, info):
            return create_response_error('create_all_speaker_preset_file',
                                         'ERROR: uploader all speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('create_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERRROR: IPC Service unreachable: {e.get_dbus_name()}'


def upload_all_speaker_preset_file(path, preset, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.upload_all_speaker_preset_file(path, preset):
            return create_response_error('upload_all_speaker_preset_file',
                                         'ERROR: uploader speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('upload_all_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERRROR: IPC Service unreachable: {e.get_dbus_name()}'

def edit_all_speaker_preset_file(path, info, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.edit_all_speaker_preset_file(path, info):
            return create_response_error('edit_all_speaker_preset_file',
                                         'ERROR: editing all speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('edit_all_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def rename_all_speaker_preset_file(path, new_file_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.rename_all_speaker_preset_file(path, new_file_name):
            return create_response_error('rename_all_speaker_preset_file',
                                         'ERROR: renaming all speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('rename_all_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def rename_all_speaker_preset_folder(folder_name, new_folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.rename_all_speaker_preset_folder(folder_name, new_folder_name):
            return create_response_error('rename_all_speaker_preset_folder',
                                         'ERROR: renaming all speaker preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('rename_all_speaker_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def create_global_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.create_global_preset_file(path):
            return create_response_error('create_global_preset_file',
                                         'ERROR: creating global preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('create_global_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def edit_global_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.edit_global_preset_file(path):
            return create_response_error('edit_global_preset_file',
                                         'ERROR: editing global preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('edit_global_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def rename_global_preset_file(path, new_file_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.rename_global_preset_file(path, new_file_name):
            return create_response_error('rename_global_preset_file',
                                         'ERROR: renaming global preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('rename_global_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def rename_global_preset_folder(folder_name, new_folder_name, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.rename_global_preset_folder(folder_name, new_folder_name):
            return create_response_error('rename_global_preset_folder',
                                         'ERROR: renaming global preset folder failed.',
                                         timestamp)
        else:
            return create_response_ok('rename_global_preset_folder', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def upload_global_preset_file(path, preset, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.upload_global_preset_file(path, preset):
            return create_response_error('upload_global_preset_file',
                                         'ERROR: uploader global preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('upload_global_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERRROR: IPC Service unreachable: {e.get_dbus_name()}'


def delete_speaker_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.delete_speaker_preset_file(path):
            return create_response_error('delete_speaker_preset_file',
                                         'ERROR: delete speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('delete_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def delete_all_speaker_preset_file(path, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.delete_all_speaker_preset_file(path):
            return create_response_error('delete_all_speaker_preset_file',
                                         'ERROR: delete all speaker preset file failed.',
                                         timestamp)
        else:
            return create_response_ok('delete_all_speaker_preset_file', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_neighbour_list(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.list_neighbours())
        if data is None:
            return create_response_error('get_neighbour_list',
                                         'ERROR: Obtaining list of neighbours failed.',
                                         timestamp)
        else:
            return create_response_ok('get_neighbour_list', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def clear_neighbour_list(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.clear_neighbours():
            return create_response_error('clear_neighbour_list',
                                         'ERROR: Clearing list of neighbours failed.',
                                         timestamp)
        else:
            return create_response_ok('clear_neighbour_list', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_local_description(desc, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.set_local_description(desc):
            return create_response_error('set_local_description',
                                         'ERROR: Failed to set local description.',
                                         timestamp)
        else:
            return create_response_ok('set_local_description', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_gain_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_gain_get(channel))
        if data is None:
            return create_response_error('get_speaker_gain',
                                         'ERROR: data obtaining for speaker gain failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_polarity_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_polarity_get(channel))
        if data is None:
            return create_response_error('get_speaker_polarity',
                                         'ERROR: data obtaining for speaker polarity failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_polarity', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_delay_enable_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_delay_enable_get(channel))
        if data is None:
            return create_response_error('get_speaker_delay_enable',
                                         'ERROR: data obtaining for speaker delay enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_delay_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_delay_value_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_delay_value_get(channel))
        if data is None:
            return create_response_error('get_speaker_delay_value',
                                         'ERROR: data obtaining for speaker delay value failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_delay_value', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_fir_table_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_fir_table_get(channel))
        if data is None:
            return create_response_error('get_speaker_fir_table',
                                         'ERROR: data obtaining for speaker fir table failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_fir_table', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_hp_enable_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_hp_enable_get(channel))
        if data is None:
            return create_response_error('get_speaker_xover_hp_enable',
                                         'ERROR: data obtaining for speaker xover high pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_xover_hp_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_hp_type_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_hp_type_get(channel))
        if data is None:
            return create_response_error('get_speaker_xover_hp_type',
                                         'ERROR: data obtaining for speaker xover high pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_xover_hp_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_hp_freq_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_hp_freq_get(channel))
        if data is None:
            return create_response_error('get_speaker_xover_hp_freq',
                                         'ERROR: data obtaining for speaker xover high pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_xover_hp_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_lp_enable_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_lp_enable_get(channel))
        if data is None:
            return create_response_error('get_speaker_xover_lp_enable',
                                         'ERROR: data obtaining for speaker xover low pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_xover_lp_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_lp_type_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_lp_type_get(channel))
        if data is None:
            return create_response_error('get_speaker_xover_lp_type',
                                         'ERROR: data obtaining for speaker xover low pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_xover_lp_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_lp_freq_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_lp_freq_get(channel))
        if data is None:
            return create_response_error('get_speaker_xover_lp_freq',
                                         'ERROR: data obtaining for speaker xover low pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_xover_lp_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_xover_hp_freq_from_preset(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_hp_freq_get_from_preset(channel))
        if data is None:
            return create_response_error('get_xover_hp_freq_from_preset',
                                         'ERROR: data obtaining for speaker xover high pass frequency from preset failed.',
                                         timestamp)
        else:
            return create_response_ok('get_xover_hp_freq_from_preset', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_xover_lp_freq_from_preset(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_xover_lp_freq_get_from_preset(channel))
        if data is None:
            return create_response_error('get_xover_lp_freq_from_preset',
                                         'ERROR: data obtaining for speaker xover low pass frequency from preset failed.',
                                         timestamp)
        else:
            return create_response_ok('get_xover_lp_freq_from_preset', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_rms_thres_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_lim_rms_thres_get(channel))
        if data is None:
            return create_response_error('get_speaker_lim_rms_thres',
                                         'ERROR: data obtaining for speaker lim rms threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_lim_rms_thres', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_rms_attack_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_lim_rms_attack_get(channel))
        if data is None:
            return create_response_error('get_speaker_lim_rms_attack',
                                         'ERROR: data obtaining for speaker lim rms attack failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_lim_rms_attack', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_rms_release_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_lim_rms_release_get(channel))
        if data is None:
            return create_response_error('get_speaker_lim_rms_release',
                                         'ERROR: data obtaining for speaker lim rms release failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_lim_rms_release', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_peak_thres_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_lim_peak_thres_get(channel))
        if data is None:
            return create_response_error('get_speaker_lim_peak_thres',
                                         'ERROR: data obtaining for speaker lim peak threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_lim_peak_thres', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_peak_release_get(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_lim_peak_release_get(channel))
        if data is None:
            return create_response_error('get_speaker_lim_peak_release',
                                         'ERROR: data obtaining for speaker lim peak release failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_lim_peak_release', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_enable_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_peq_enable_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_speaker_peq_enable',
                                         'ERROR: data obtaining for speaker peq enable failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_peq_enable', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_type_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_peq_type_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_speaker_peq_type',
                                         'ERROR: data obtaining for speaker peq type failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_peq_type', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_freq_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_peq_freq_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_speaker_peq_freq',
                                         'ERROR: data obtaining for speaker peq freq failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_peq_freq', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_gain_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_peq_gain_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_speaker_peq_gain',
                                         'ERROR: data obtaining for speaker peq gain failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_peq_gain', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_q_get(channel, peq_channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.speaker_peq_q_get(channel, peq_channel))
        if data is None:
            return create_response_error('get_speaker_peq_q',
                                         'ERROR: data obtaining for speaker peq q failed.',
                                         timestamp)
        else:
            return create_response_ok('get_speaker_peq_q', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_gain_set(channel, gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_gain_set(channel, gain):
            return create_response_error('set_speaker_gain', 'ERROR: setting speaker gain failed.', timestamp)
        else:
            return create_response_ok('set_speaker_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_mute_set(channel, muted, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_mute_set(channel, muted):
            return create_response_error('set_speaker_mute', 'ERROR: setting speaker mute failed.', timestamp)
        else:
            return create_response_ok('set_speaker_mute', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_polarity_set(channel, polarity, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_polarity_set(channel, polarity):
            return create_response_error('set_speaker_polarity', 'ERROR: setting speaker polarity failed.', timestamp)
        else:
            return create_response_ok('set_speaker_polarity', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_delay_enable_set(channel, delay_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_delay_enable_set(channel, delay_enabled):
            return create_response_error('set_speaker_delay_enable',
                                         'ERROR: setting speaker delay enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_delay_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_delay_value_set(channel, delay_value, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_delay_value_set(channel, delay_value):
            return create_response_error('set_speaker_delay_value',
                                         'ERROR: setting speaker delay value failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_delay_value', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_fir_table_set(channel, fir_table, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_fir_table_set(channel, fir_table):
            return create_response_error('set_speaker_fir_table', 'ERROR: setting speaker fir table failed.', timestamp)
        else:
            return create_response_ok('set_speaker_fir_table', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_hp_enable_set(channel, xover_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_hp_enable_set(channel, xover_enabled):
            return create_response_error('set_speaker_xover_hp_enable',
                                         'ERROR: setting speaker xover high pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_hp_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_hp_type_set(channel, xover_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_hp_type_set(channel, xover_type):
            return create_response_error('set_speaker_xover_hp_type',
                                         'ERROR: setting speaker high pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_hp_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_hp_freq_set(channel, xover_freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_hp_freq_set(channel, xover_freq):
            return create_response_error('set_speaker_xover_hp_freq',
                                         'ERROR: setting speaker xover high pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_hp_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def speaker_xover_hp_overide_set(channel, overide_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_hp_overide_set(channel, overide_enabled):
            return create_response_error('set_speaker_xover_hp_overide',
                                         'ERROR: setting speaker xover high pass overide enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_hp_overide', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def speaker_xover_lp_overide_set(channel, overide_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_lp_overide_set(channel, overide_enabled):
            return create_response_error('set_speaker_xover_lp_overide',
                                         'ERROR: setting speaker xover low pass overide enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_lp_overide', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def speaker_xover_lp_enable_set(channel, xover_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_lp_enable_set(channel, xover_enabled):
            return create_response_error('set_speaker_xover_lp_enable',
                                         'ERROR: setting speaker xover low pass enable failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_lp_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_lp_type_set(channel, xover_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_lp_type_set(channel, xover_type):
            return create_response_error('set_speaker_xover_lp_type',
                                         'ERROR: setting speaker xover low pass type failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_lp_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_xover_lp_freq_set(channel, xover_freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_xover_lp_freq_set(channel, xover_freq):
            return create_response_error('set_speaker_xover_lp_freq',
                                         'ERROR: setting speaker xover low pass frequency failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_xover_lp_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_rms_thres_set(channel, thres, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_lim_rms_thres_set(channel, thres):
            return create_response_error('set_speaker_lim_rms_thres',
                                         'ERROR: setting speaker lim rms threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_lim_rms_thres', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_rms_attack_set(channel, attack, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_lim_rms_attack_set(channel, attack):
            return create_response_error('set_speaker_lim_rms_attack',
                                         'ERROR: setting speaker lim rms attack failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_lim_rms_attack', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_rms_release_set(channel, release, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_lim_rms_release_set(channel, release):
            return create_response_error('set_speaker_lim_rms_release',
                                         'ERROR: setting speaker lim rms release failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_lim_rms_release', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_peak_thres_set(channel, thres, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_lim_peak_thres_set(channel, thres):
            return create_response_error('set_speaker_lim_peak_thres',
                                         'ERROR: setting speaker lim peak threshold failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_lim_peak_thres', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_lim_peak_release_set(channel, release, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_lim_peak_release_set(channel, release):
            return create_response_error('set_speaker_lim_peak_release',
                                         'ERROR: setting speaker lim peak release failed.',
                                         timestamp)
        else:
            return create_response_ok('set_speaker_lim_peak_release', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_enable_set(channel, peq_channel, peq_enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_peq_enable_set(channel, peq_channel, peq_enabled):
            return create_response_error('set_speaker_peq_enable', 'ERROR: setting speaker peq enable failed.', timestamp)
        else:
            return create_response_ok('set_speaker_peq_enable', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_type_set(channel, peq_channel, peq_type, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_peq_type_set(channel, peq_channel, peq_type):
            return create_response_error('set_speaker_peq_type', 'ERROR: setting speaker peq type failed.', timestamp)
        else:
            return create_response_ok('set_speaker_peq_type', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_freq_set(channel, peq_channel, peq_freq, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_peq_freq_set(channel, peq_channel, peq_freq):
            return create_response_error('set_speaker_peq_freq', 'ERROR: setting speaker peq frequency failed.', timestamp)
        else:
            return create_response_ok('set_speaker_peq_freq', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_gain_set(channel, peq_channel, peq_gain, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_peq_gain_set(channel, peq_channel, peq_gain):
            return create_response_error('set_speaker_peq_gain', 'ERROR: setting speaker peq gain failed.', timestamp)
        else:
            return create_response_ok('set_speaker_peq_gain', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_12(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_12_get())
        if data is None:
            return create_response_error('get_bridge_mode_12',
                                         'ERROR: data obtaining for bridge mode 34 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_12', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_34(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_34_get())
        if data is None:
            return create_response_error('get_bridge_mode_34',
                                         'ERROR: data obtaining for bridge mode 34 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_34', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_56(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_56_get())
        if data is None:
            return create_response_error('get_bridge_mode_56',
                                         'ERROR: data obtaining for bridge mode 56 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_56', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_78(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_78_get())
        if data is None:
            return create_response_error('get_bridge_mode_78',
                                         'ERROR: data obtaining for bridge mode 78 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_78', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_910(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_910_get())
        if data is None:
            return create_response_error('get_bridge_mode_910',
                                         'ERROR: data obtaining for bridge mode 910 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_910', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_1112(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_1112_get())
        if data is None:
            return create_response_error('get_bridge_mode_1112',
                                         'ERROR: data obtaining for bridge mode 1112 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_1112', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_1314(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_1314_get())
        if data is None:
            return create_response_error('get_bridge_mode_1314',
                                         'ERROR: data obtaining for bridge mode 1314 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_1314', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_bridge_mode_1516(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.bridge_mode_1516_get())
        if data is None:
            return create_response_error('get_bridge_mode_1516',
                                         'ERROR: data obtaining for bridge mode 1516 failed.',
                                         timestamp)
        else:
            return create_response_ok('get_bridge_mode_1516', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_12(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_12_set(enabled):
            return create_response_error('set_bridge_mode_12', 'ERROR: setting bridge mode 12 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_12', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_34(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_34_set(enabled):
            return create_response_error('set_bridge_mode_34', 'ERROR: setting bridge mode 34 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_34', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_56(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_56_set(enabled):
            return create_response_error('set_bridge_mode_56', 'ERROR: setting bridge mode 56 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_56', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_78(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_78_set(enabled):
            return create_response_error('set_bridge_mode_78', 'ERROR: setting bridge mode 78 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_78', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_910(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_910_set(enabled):
            return create_response_error('set_bridge_mode_910', 'ERROR: setting bridge mode 910 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_910', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_1112(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_1112_set(enabled):
            return create_response_error('set_bridge_mode_1112', 'ERROR: setting bridge mode 1112 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_1112', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_1314(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_1314_set(enabled):
            return create_response_error('set_bridge_mode_1314', 'ERROR: setting bridge mode 1314 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_1314', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_bridge_mode_1516(enabled, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.bridge_mode_1516_set(enabled):
            return create_response_error('set_bridge_mode_1516', 'ERROR: setting bridge mode 1516 failed.', timestamp)
        else:
            return create_response_ok('set_bridge_mode_1516', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def speaker_peq_q_set(channel, peq_channel, peq_q, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.speaker_peq_q_set(channel, peq_channel, peq_q):
            return create_response_error('set_speaker_peq_q', 'ERROR: setting speaker peq q failed.', timestamp)
        else:
            return create_response_ok('set_speaker_peq_q', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_lan_config(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.ethernet_get_config())
        if data is None:
            return create_response_error('get_lan_config',
                                         'ERROR: data obtaining for lan config failed.',
                                         timestamp)
        else:
            return create_response_ok('get_lan_config', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_apex_cloud(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.apex_cloud_get())
        if data is None:
            return create_response_error('get_apex_cloud',
                                         'ERROR: data obtaining for apex cloud failed.',
                                         timestamp)
        else:
            return create_response_ok('get_apex_cloud', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_spotify_config(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.spotify_config_get())
        if data is None:
            return create_response_error('get_spotify_config',
                                         'ERROR: data obtaining for spotify config failed.',
                                         timestamp)
        else:
            return create_response_ok('get_spotify_config', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_airplay_config(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.airplay_config_get())
        if data is None:
            return create_response_error('get_airplay_config',
                                         'ERROR: data obtaining for airplay config failed.',
                                         timestamp)
        else:
            return create_response_ok('get_airplay_config', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_hotspot_config(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.hotspot_config_get())
        if data is None:
            return create_response_error('get_hotspot_config',
                                         'ERROR: data obtaining for hotspot config failed.',
                                         timestamp)
        else:
            return create_response_ok('get_hotspot_config', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_about(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.about_get())
        if data is None:
            return create_response_error('get_about',
                                         'ERROR: data obtaining for about failed.',
                                         timestamp)
        else:
            return create_response_ok('get_about', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_status(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.status_get())
        if data is None:
            return create_response_error('get_status',
                                         'ERROR: data obtaining for status failed.',
                                         timestamp)
        else:
            return create_response_ok('get_status', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_apex_cloud(data, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.apex_cloud_set(data):
            return create_response_error('set_apex_cloud', 'ERROR: setting apex cloud failed.', timestamp)
        else:
            return create_response_ok('set_apex_cloud', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_spotify_config(config, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.spotify_config_set(config):
            return create_response_error('set_spotify_config', 'ERROR: setting spotify config failed.', timestamp)
        else:
            return create_response_ok('set_spotify_config', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def set_airplay_config(data, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.airplay_config_set(data):
            return create_response_error('set_airplay_config', 'ERROR: setting airplay failed.', timestamp)
        else:
            return create_response_ok('set_airplay_config', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'


def get_json_data(keypaths, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.get_json_data(keypaths))
        if data is None or data == '<ERROR>':
            return create_response_error('get_json_data',
                                         'ERROR: Getting data failed.',
                                         timestamp)
        else:
            return create_response_ok('get_json_data', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def oem_check_code(code, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.oem_check_code(code))
        if data:
            return create_response_ok('oem_check_code', timestamp)
        else:
            return create_response_error('oem_check_code', 'KO', timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def oem_set_code(label, code, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.oem_set_code(label, code):
            return create_response_error('oem_set_code', 'ERROR: set oem code failed', timestamp)
        else:
            return create_response_ok('oem_set_code', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def oem_remove_code(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.oem_remove_code():
            return create_response_error('oem_remove_code', 'ERROR: remove oem code failed', timestamp)
        else:
            return create_response_ok('oem_remove_code', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def oem_get_status(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.oem_get_status())
        if data is None:
            return create_response_error('oem_get_status',
                                         'ERROR: data obtaining for oem status failed.',
                                         timestamp)
        else:
            return create_response_ok('oem_get_status', data, timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def hiz_set_filter(channel, filter, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.hiz_filter_set(channel, filter):
            return create_response_error('set_hiz_filter',
                                         'ERROR: unable to set hiz filter.',
                                         timestamp)
        else:
            return create_response_ok('set_hiz_filter', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def hiz_get_filter(channel, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.hiz_filter_get(channel):
            return create_response_error('get_hiz_filter',
                                         'ERROR: unable to get hiz filter.',
                                         timestamp)
        else:
            return create_response_ok('get_hiz_filter', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def front_lock_set(lock, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.front_lock_set(lock):
            return create_response_error('set_front_lock',
                                         'ERROR: unable to set front lock.',
                                         timestamp)
        else:
            return create_response_ok('set_front_lock', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def front_lock_get(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.front_lock_get())
        if data is None:
            return create_response_error('get_front_lock',
                                         'ERROR: unable to get front lock.',
                                         timestamp)
        else:
            return create_response_ok('get_front_lock', bool(data), timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'

def locate_run(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.locate_run():
            return create_response_error('locate_run',
                                         'ERROR: unable to run locate.',
                                         timestamp)
        else:
            return create_response_ok('locate_run', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    
def factory_reset(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.factory_reset():
            return create_response_error('factory_reset',
                                         'ERROR: unable to reset to factory.',
                                         timestamp)
        else:
            return create_response_ok('factory_reset', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    
def audio_reset(timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        if not proxy_object.audio_reset():
            return create_response_error('audio_reset',
                                         'ERROR: unable to reset to factory.',
                                         timestamp)
        else:
            return create_response_ok('audio_reset', timestamp=timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}'
    
def copy_channel_parameters_set(source, destination, copy, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.copy_channel_parameters_set(source, destination, copy))
        if data is True:
            return create_response_ok('set_copy_channel_parameters', timestamp=timestamp)
        else:
            return create_response_error('set_copy_channel_parameters',
                                         'ERROR: unable to copy channel parameters.',
                                         timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}' + ' // ' + source + ' // ' + destination +  ' // ' + copy
    
def copy_speaker_parameters_set(source, destination, copy, timestamp):
    try:
        proxy_object = dbus.SystemBus().get_object(ipc_common.SERVICE_BUS, ipc_common.SERVICE_OBJECT_PATH)
        data = ipc_util.dbus_to_python(proxy_object.copy_speaker_parameters_set(source, destination, copy))
        if data is True:
            return create_response_ok('set_copy_preset_parameters', timestamp=timestamp)
        else:
            return create_response_error('set_copy_preset_parameters',
                                         'ERROR: unable to copy preset parameters.',
                                         timestamp)
    except dbus.DBusException as e:
        return f'ERROR: IPC Service unreachable: {e.get_dbus_name()}' + ' // ' + source + ' // ' + destination +  ' // ' + copy

def create_response_ok(command_id, value='', timestamp=''):
    if '' == value:
        if '' == timestamp:
            response = {"command": command_id, "arg1": "OK"}
        else:
            response = {"command": command_id, "arg1": "OK", "query_time": timestamp}
    else:
        if '' == timestamp:
            response = {"command": command_id, "arg1": "OK", "arg2": value}
        else:
            response = {"command": command_id, "arg1": "OK", "arg2": value, "query_time": timestamp}
    return json.dumps(response)


def create_response_error(command_id, error_message, timestamp=''):
    if '' == timestamp:
        response = {"command": command_id, "arg1": error_message}
    else:
        response = {"command": command_id, "arg1": error_message, "query_time": timestamp}
    return json.dumps(response)


def main(exe, args):
    if sys.version_info[0] < 3:
        sys.exit("Must run using Python 3 or newer.")

    # One-argument commands:
    if len(args) == 1:
        if args[0] == 'eth_dhcp':
            print(eth_config(None, ipc_util.get_timestamp()))
        elif args[0] == 'protocol_test':
            print(protocol_test(ipc_util.get_timestamp()))
        elif args[0] == 'readback_start':
            print(readback_ctl(True, ipc_util.get_timestamp()))
        elif args[0] == 'readback_stop':
            print(readback_ctl(False, ipc_util.get_timestamp()))
        elif args[0] == 'readback_get':
            print(readback_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_home_page_status':
            print(data_home_page_status_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_audio_delay':
            print(data_audio_delay_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_audio_gain':
            print(data_audio_gain_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_audio_source':
            print(data_audio_source_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_global_preset_folder_names':
            print(global_preset_folder_names_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_preset_folder_names':
            print(speaker_preset_folder_names_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_hotspot':
            print(data_hotspot_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_cloud':
            print(data_cloud_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_network':
            print(data_network_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_data_about':
            print(data_about_get(ipc_util.get_timestamp()))
        elif args[0] == 'get_neighbour_list':
            print(get_neighbour_list(ipc_util.get_timestamp()))
        elif args[0] == 'clear_neighbour_list':
            print(clear_neighbour_list(ipc_util.get_timestamp()))
        elif args[0] == 'get_bridge_mode_12':
            print(get_bridge_mode_12(ipc_util.get_timestamp()))
        elif args[0] == 'get_bridge_mode_34':
            print(get_bridge_mode_34(ipc_util.get_timestamp()))
        elif args[0] == 'get_lan_config':
            print(get_lan_config(ipc_util.get_timestamp()))
        elif args[0] == 'get_apex_cloud':
            print(get_apex_cloud(ipc_util.get_timestamp()))
        elif args[0] == 'get_spotify_config':
            print(get_spotify_config(ipc_util.get_timestamp()))
        elif args[0] == 'get_airplay_config':
            print(get_airplay_config(ipc_util.get_timestamp()))
        elif args[0] == 'get_hotspot_config':
            print(get_hotspot_config(ipc_util.get_timestamp()))
        elif args[0] == 'get_about':
            print(get_about(ipc_util.get_timestamp()))
        else:
            print("One-argument command: Unsupported command.")

    # Two-argument commands:
    elif len(args) == 2:
        if args[0] == 'get_json_data':
            print(get_json_data(args[1], ipc_util.get_timestamp()))
        elif args[0] == 'eth_static':
            print(eth_config(args[1], ipc_util.get_timestamp()))
        elif args[0] == 'hotspot_enable':
            print(hotspot_enable(bool(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'set_hotspot_config':
            print(set_hotspot_config(args[1], ipc_util.get_timestamp()))
        elif args[0] == 'cloud_enable':
            print(cloud_enable(bool(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'global_preset_set':
            print(global_preset_set(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_global_preset_file_names':
            print(global_preset_file_names_get(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_preset_file_names':
            print(speaker_preset_file_names_get(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_mute':
            print(solo_mute_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_gain':
            print(solo_gain_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_polarity':
            print(solo_polarity_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_delay_enable':
            print(solo_delay_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_delay_value':
            print(solo_delay_value_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_xover_hp_enable':
            print(solo_xover_hp_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_xover_hp_type':
            print(solo_xover_hp_type_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_xover_hp_freq':
            print(solo_xover_hp_freq_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_xover_lp_enable':
            print(solo_xover_lp_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_xover_lp_type':
            print(solo_xover_lp_type_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_xover_lp_freq':
            print(solo_xover_lp_freq_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_lim_rms_thres':
            print(solo_lim_rms_thres_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_lim_rms_attack':
            print(solo_lim_rms_attack_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_lim_rms_release':
            print(solo_lim_rms_release_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_lim_peak_thres':
            print(solo_lim_peak_thres_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_lim_peak_release':
            print(solo_lim_peak_release_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_channel_source_pri':
            print(solo_channel_source_pri_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_channel_source_sec':
            print(solo_channel_source_sec_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_channel_name':
            print(solo_channel_name_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_mute':
            print(group_mute_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_gain':
            print(group_gain_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_polarity':
            print(group_polarity_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_delay_enable':
            print(group_delay_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_delay_value':
            print(group_delay_value_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_name':
            print(group_name_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_assign_channel_to_group':
            print(assign_channel_to_group_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_preset_file':
            print(ipc_util.dbus_to_python(speaker_preset_file_get(str(args[1]), ipc_util.get_timestamp())))
        elif args[0] == 'get_global_preset_file':
            print(ipc_util.dbus_to_python(global_preset_file_get(str(args[1]), ipc_util.get_timestamp())))
        elif args[0] == 'delete_speaker_preset_file':
            print(delete_speaker_preset_file(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'delete_global_preset_file':
            print(delete_global_preset_file(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'create_speaker_preset_folder':
            print(create_speaker_preset_folder(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'create_global_preset_folder':
            print(create_global_preset_folder(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'delete_speaker_preset_folder':
            print(delete_speaker_preset_folder(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'delete_global_preset_folder':
            print(delete_global_preset_folder(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_gain':
            print(speaker_gain_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_polarity':
            print(speaker_polarity_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_delay_enable':
            print(speaker_delay_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_delay_value':
            print(speaker_delay_value_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_fir_table':
            print(speaker_fir_table_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_xover_hp_enable':
            print(speaker_xover_hp_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_xover_hp_type':
            print(speaker_xover_hp_type_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_xover_hp_freq':
            print(speaker_xover_hp_freq_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_xover_lp_enable':
            print(speaker_xover_lp_enable_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_xover_lp_type':
            print(speaker_xover_lp_type_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_xover_lp_freq':
            print(speaker_xover_lp_freq_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_lim_rms_thres':
            print(speaker_lim_rms_thres_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_lim_rms_attack':
            print(speaker_lim_rms_attack_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_lim_rms_release':
            print(speaker_lim_rms_release_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_lim_peak_thres':
            print(speaker_lim_peak_thres_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_lim_peak_release':
            print(speaker_lim_peak_release_get(int(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'set_local_description':
            print(set_local_description(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'create_global_preset_file':
            print(create_global_preset_file(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'edit_global_preset_file':
            print(edit_global_preset_file(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'set_global_preset_file':
            print(set_global_preset_file(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'set_bridge_mode_12':
            print(set_bridge_mode_12(ipc_util.convert_str_to_dbus_msg(str(args[1])), ipc_util.get_timestamp()))
        elif args[0] == 'set_bridge_mode_34':
            print(set_bridge_mode_34(ipc_util.convert_str_to_dbus_msg(str(args[1])), ipc_util.get_timestamp()))
        elif args[0] == 'set_apex_cloud':
            print(set_apex_cloud(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'set_spotify_config':
            print(set_spotify_config(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'set_airplay_config':
            print(set_airplay_config(str(args[1]), ipc_util.get_timestamp()))
        elif args[0] == 'get_hiz_player':
            print(hiz_get_filter(int(args[1]), ipc_util.get_timestamp()))
        else:
            print("Two-argument command: Unsupported command.")

    # Three-argument commands:
    elif len(args) == 3:
        if args[0] == 'speaker_preset_set':
            print(speaker_preset_set(int(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_peq_enable':
            print(solo_peq_enable_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_peq_type':
            print(solo_peq_type_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_peq_freq':
            print(solo_peq_freq_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_peq_gain':
            print(solo_peq_gain_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_solo_peq_q':
            print(solo_peq_q_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_mute':
            print(solo_mute_set(int(args[1]), ipc_util.convert_str_to_dbus_msg(str(args[2])), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_gain':
            print(solo_gain_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_polarity':
            print(solo_polarity_set(int(args[1]),
                                    ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                    ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_delay_enable':
            print(solo_delay_enable_set(int(args[1]),
                                        ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                        ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_delay_value':
            print(solo_delay_value_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_xover_hp_enable':
            print(solo_xover_hp_enable_set(int(args[1]),
                                           ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                           ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_xover_hp_type':
            print(solo_xover_hp_type_set(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_xover_hp_freq':
            print(solo_xover_hp_freq_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_xover_lp_enable':
            print(solo_xover_lp_enable_set(int(args[1]),
                                           ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                           ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_xover_lp_type':
            print(solo_xover_lp_type_set(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_xover_lp_freq':
            print(solo_xover_lp_freq_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_lim_rms_thres':
            print(solo_lim_rms_thres_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_lim_rms_attack':
            print(solo_lim_rms_attack_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_lim_rms_release':
            print(solo_lim_rms_release_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_lim_peak_thres':
            print(solo_lim_peak_thres_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_lim_peak_release':
            print(solo_lim_peak_release_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_channel_source_pri':
            print(solo_channel_source_pri_set(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_channel_source_sec':
            print(solo_channel_source_sec_set(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_channel_name':
            print(solo_channel_name_set(int(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_peq_enable':
            print(group_peq_enable_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_peq_type':
            print(group_peq_type_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_peq_freq':
            print(group_peq_freq_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_peq_gain':
            print(group_peq_gain_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_group_peq_q':
            print(group_peq_q_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_mute':
            print(group_mute_set(int(args[1]),
                                 ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                 ipc_util.get_timestamp()))
        elif args[0] == 'set_group_gain':
            print(group_gain_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_polarity':
            print(group_polarity_set(int(args[1]),
                                     ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                     ipc_util.get_timestamp()))
        elif args[0] == 'set_group_delay_enable':
            print(group_delay_enable_set(int(args[1]),
                                         ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                         ipc_util.get_timestamp()))
        elif args[0] == 'set_group_delay_value':
            print(group_delay_value_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_name':
            print(group_name_set(int(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'rename_speaker_preset_file':
            print(rename_speaker_preset_file(str(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'rename_global_preset_file':
            print(rename_global_preset_file(str(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'rename_speaker_preset_folder':
            print(rename_speaker_preset_folder(str(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'rename_global_preset_folder':
            print(rename_global_preset_folder(str(args[1]), str(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_peq_enable':
            print(speaker_peq_enable_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_peq_type':
            print(speaker_peq_type_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_peq_freq':
            print(speaker_peq_freq_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_peq_gain':
            print(speaker_peq_gain_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'get_speaker_peq_q':
            print(speaker_peq_q_get(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_gain':
            print(speaker_gain_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_polarity':
            print(speaker_polarity_set(int(args[1]),
                                       ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                       ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_delay_enable':
            print(speaker_delay_enable_set(int(args[1]),
                                           ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                           ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_delay_value':
            print(speaker_delay_value_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_fir_table':
            print(speaker_fir_table_set(int(args[1]),
                                        ipc_util.python_to_dbus([1, 2, 3, 4, 5]),
                                        ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_xover_hp_enable':
            print(speaker_xover_hp_enable_set(int(args[1]),
                                              ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                              ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_xover_hp_type':
            print(speaker_xover_hp_type_set(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_xover_hp_freq':
            print(speaker_xover_hp_freq_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_xover_lp_enable':
            print(speaker_xover_lp_enable_set(int(args[1]),
                                              ipc_util.convert_str_to_dbus_msg(str(args[2])),
                                              ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_xover_lp_type':
            print(speaker_xover_lp_type_set(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_xover_lp_freq':
            print(speaker_xover_lp_freq_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_lim_rms_thres':
            print(speaker_lim_rms_thres_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_lim_rms_attack':
            print(speaker_lim_rms_attack_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_lim_rms_release':
            print(speaker_lim_rms_release_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_lim_peak_thres':
            print(speaker_lim_peak_thres_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_lim_peak_release':
            print(speaker_lim_peak_release_set(int(args[1]), float(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'edit_speaker_preset_file':
            print(edit_speaker_preset_file(str(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_preset_file':
            print(set_speaker_preset_file(str(args[1]), int(args[2]), ipc_util.get_timestamp()))
        elif args[0] == 'set_hiz_player':
            print(hiz_set_filter(int(args[1]), int(args[2]), ipc_util.get_timestamp()))
        else:
            print("Three-argument command: Unsupported command.")

    # Four-argument commands:
    elif len(args) == 4:
        if args[0] == 'set_solo_peq_enable':
            print(solo_peq_enable_set(int(args[1]),
                                      int(args[2]),
                                      ipc_util.convert_str_to_dbus_msg(str(args[3])),
                                      ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_peq_type':
            print(solo_peq_type_set(int(args[1]), int(args[2]), int(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_peq_freq':
            print(solo_peq_freq_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_peq_gain':
            print(solo_peq_gain_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_solo_peq_q':
            print(solo_peq_q_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_peq_enable':
            print(group_peq_enable_set(int(args[1]),
                                       int(args[2]),
                                       ipc_util.convert_str_to_dbus_msg(str(args[3])),
                                       ipc_util.get_timestamp()))
        elif args[0] == 'set_group_peq_type':
            print(group_peq_type_set(int(args[1]), int(args[2]), int(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_peq_freq':
            print(group_peq_freq_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_peq_gain':
            print(group_peq_gain_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_group_peq_q':
            print(group_peq_q_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_assign_channel_to_group':
            print(assign_channel_to_group_set(int(args[1]), int(args[2]), int(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_peq_enable':
            print(speaker_peq_enable_set(int(args[1]),
                                         int(args[2]),
                                         ipc_util.convert_str_to_dbus_msg(str(args[3])),
                                         ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_peq_type':
            print(speaker_peq_type_set(int(args[1]), int(args[2]), int(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_peq_freq':
            print(speaker_peq_freq_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_peq_gain':
            print(speaker_peq_gain_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'set_speaker_peq_q':
            print(speaker_peq_q_set(int(args[1]), int(args[2]), float(args[3]), ipc_util.get_timestamp()))
        elif args[0] == 'create_speaker_preset_file':
            print(create_speaker_preset_file(str(args[1]), int(args[2]), str(args[3]), ipc_util.get_timestamp()))
        else:
            print("Four-argument command: Unsupported command.")

    # Unsupported commands:
    else:
        sys.exit('Usage: %s command' % exe)


if __name__ == '__main__':
    main(sys.argv[0], sys.argv[1:])
